//------------------------------------------------------------------------
//  LEVEL building - QUAKE 1 BSP
//------------------------------------------------------------------------
//
//  Oblige Level Maker
//
//  Copyright (C) 2006-2009 Andrew Apted
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//------------------------------------------------------------------------

#include "headers.h"
#include "hdr_fltk.h"
#include "hdr_lua.h"
#include "hdr_ui.h"

#include "lib_file.h"
#include "lib_util.h"
#include "main.h"

#include "csg_main.h"

#include "g_lua.h"

#include "q_bsp.h"
#include "q1_main.h"
#include "q1_structs.h"


#define FACE_MAX_SIZE  240


extern bool CSG2_PointInSolid(double x, double y);


class qSide_c;
class qLeaf_c;
class qNode_c;


static qNode_c *Q_ROOT;

static qLeaf_c *SOLID_LEAF;


void DoAssignFaces(qNode_c *N, qSide_c *S);


class qFace_c
{
public:
  enum face_kind_e
  {
    WALL  = 0,
    FLOOR = 1,
    CEIL  = 2
  };

  int kind;

  double z1, z2;

  int gap;

  qSide_c *side;
  qLeaf_c *floor_leaf;

  int index;  // final index into Faces lump

public:
   qFace_c(int _kind = WALL) : kind(_kind), side(NULL), floor_leaf(NULL), index(-1) { }
  ~qFace_c() { }

   qFace_c(int _kind, int _gap, double _z1, double _z2) :
           kind(_kind), z1(_z1), z2(_z2), gap(_gap),
           side(NULL), floor_leaf(NULL), index(-1)
   { }

   qFace_c(const qFace_c *other, qSide_c *new_side) :
           kind(other->kind), z1(other->z1), z2(other->z2),
           gap(other->gap), side(new_side),
           floor_leaf(NULL), index(-1)
   { }
};


class qSide_c
{
public:
  merge_segment_c *seg;  // NULL means "portal"

  int side;  // 0 is front, 1 is back

  double x1, y1;
  double x2, y2;

  // faces on this side
  std::vector<qFace_c *> faces;
 
  bool original;  // false for split-off pieces

  qNode_c * on_node;  // non-null if has been on a partition line

public:
  qSide_c(merge_segment_c * _seg, int _side) :
      seg(_seg), side(_side), faces(), original(true), on_node(NULL)
  {
    if (side == 0)
    {
      x1 = seg->start->x;  x2 = seg->end->x;
      y1 = seg->start->y;  y2 = seg->end->y;
    }
    else  // back
    {
      x1 = seg->end->x;  x2 = seg->start->x;
      y1 = seg->end->y;  y2 = seg->start->y;
    }
  }

  ~qSide_c()
  {
      // TODO: delete the faces
  }

private:
  // copy constructor, used when splitting
  qSide_c(const qSide_c *other, double new_x, double new_y) :
          seg(other->seg), side(other->side),
          x1(new_x), y1(new_y), x2(other->x2), y2(other->y2),
          faces(), original(false), on_node(other->on_node)
  { }

  // for NewPortal
  qSide_c(double px1, double py1, double px2, double py2, int _side) :
      seg(NULL), side(_side), faces(), original(true),
      on_node(NULL) // FIXME !!!!!
  {
    if (side == 0)
    {
      x1 = px1; y1 = py1;
      x2 = px2; y2 = py2;
    }
    else  // back
    {
      x1 = px2; y1 = py2;
      x2 = px1; y2 = py1;
    }
  }

public:
  double Length() const
  {
    return ComputeDist(x1,y1, x2,y2);
  }

  merge_region_c *GetRegion() const
  {
    SYS_ASSERT(seg);
    return (side == 0) ? seg->front : seg->back;
  }

  qSide_c *SplitAt(double new_x, double new_y)
  {
    qSide_c *T = new qSide_c(this, new_x, new_y);

    x2 = new_x;
    y2 = new_y;

    // duplicate the faces
    for (unsigned int i = 0; i < faces.size(); i++)
    {
      T->faces.push_back(new qFace_c(faces[i], T));
    }

    if (on_node)
      DoAssignFaces(on_node, T);

    return T;
  }

  static qSide_c *NewPortal(double px1, double py1, double px2, double py2,
                     int _side)
  {
    return new qSide_c(px1,py1, px2,py2, _side);
  }

  void AddFace(qFace_c *F)
  {
    SYS_ASSERT(original);

    faces.push_back(F);

    F->side = this;
  }
};


class qLeaf_c
{
public:
  int contents;

  std::list<qSide_c *> sides;

  // Note: qSide_c objects are shared when gap > 0

  int gap;
  int numgap;

  double min_x, min_y;
  double max_x, max_y;

  // list of faces is created when the leaf is vertically partitioned
  // NB: faces are managed by qSide_c, we only store ptr-copies here
  std::vector<qFace_c *> faces;

  qFace_c *floor;
  qFace_c *ceil;

  bool floor_on_node;
  bool ceil_on_node;

public:
  qLeaf_c() : contents(CONTENTS_EMPTY), /* faces(), */ sides(),
              gap(0), numgap(0),
              min_x(0), min_y(0), max_x(0), max_y(0),
              faces(), floor(NULL), ceil(NULL),
              floor_on_node(false), ceil_on_node(false)
  { }

  ~qLeaf_c()
  {
    // TODO: delete faces and sides
  }

  qLeaf_c(qLeaf_c& other, int _gap) :
          contents(other.contents), sides(), gap(_gap),
          min_x(other.min_x), min_y(other.min_y),
          max_x(other.max_x), max_y(other.max_y),
          faces(), floor(NULL), ceil(NULL),
          floor_on_node(false), ceil_on_node(false)
  {
    // copy the side pointers
    std::list<qSide_c *>::iterator SI;

    for (SI = other.sides.begin(); SI != other.sides.end(); SI++)
      sides.push_back(*SI);

    // we don't copy faces (???)
  }

  qSide_c * AddSide(merge_segment_c *_seg, int _side)
  {
    qSide_c *S = new qSide_c(_seg, _side); 

    sides.push_back(S);

#if 0
    fprintf(stderr, "Side #%p : seg (%1.0f,%1.0f) - (%1.0f,%1.0f) side:%d\n",
         S, _seg->start->x, _seg->start->y,
           _seg->end->x, _seg->end->y, _side);
#endif
    return S;
  }

  merge_region_c *GetRegion() // const
  {
    // NOTE: assumes a convex leaf (in XY) !!
    for (std::list<qSide_c *>::iterator SI = sides.begin();
         SI != sides.end();
         SI++)
    {
      if ((*SI)->seg)
        return (*SI)->GetRegion();
    }

    Main_FatalError("INTERNAL ERROR: Leaf %p has no solid side!", this);
    return NULL; /* NOT REACHED */
  }

  bool HasSide(qSide_c *side)
  {
    for (std::list<qSide_c *>::iterator SI = sides.begin();
         SI != sides.end();
         SI++)
    {
      if ((*SI) == side)
        return true;
    }

    return false;
  }

  merge_gap_c *GetGap()
  {
    SYS_ASSERT(numgap == 1);

    merge_region_c *R = GetRegion();

    SYS_ASSERT(R);
    SYS_ASSERT(gap >= 0 && gap < (int)R->gaps.size());

    return R->gaps[gap];
  }

  void ComputeBBox()
  {
    min_x = min_y = +9e9;
    max_x = max_y = -9e9;

    std::list<qSide_c *>::iterator SI;

    for (SI = sides.begin(); SI != sides.end(); SI++)
    {
      qSide_c *S = (*SI);

      if (S->x1 < min_x) min_x = S->x1;
      if (S->x2 < min_x) min_x = S->x2;
      if (S->y1 < min_y) min_y = S->y1;
      if (S->y2 < min_y) min_y = S->y2;

      if (S->x1 > max_x) max_x = S->x1;
      if (S->x2 > max_x) max_x = S->x2;
      if (S->y1 > max_y) max_y = S->y1;
      if (S->y2 > max_y) max_y = S->y2;
    }
  }

  void AssignFaces()
  {
    SYS_ASSERT(numgap == 1);

    std::list<qSide_c *>::iterator SI;

    for (SI = sides.begin(); SI != sides.end(); SI++)
    {
      qSide_c *S = *SI;

      if (! S->seg) // ignore portals
        continue;

      for (unsigned int k = 0; k < S->faces.size(); k++)
      {
        qFace_c *F = S->faces[k];

        // check if already there
        bool already = false;
        for (unsigned int m=0; m < faces.size(); m++)
        {
          SYS_ASSERT(faces[m] != F); // { already = true; break; }
        }

        if (F->gap == gap && !already)
          faces.push_back(F);
      }
    }
  }
};


class qNode_c
{
public:
  // true if this node splits the tree by Z
  // (with a horizontal upward-facing plane, i.e. dz = 1).
  int z_splitter;

  double z;

  // normal splitting planes are vertical, and here are the
  // coordinates on the map.
  double x,  y;
  double dx, dy;

  qLeaf_c *front_l;  // front space : one of these is non-NULL
  qNode_c *front_n;

  qLeaf_c *back_l;   // back space : one of these is non-NULL
  qNode_c *back_n;

  double min_x, min_y;
  double max_x, max_y;

  // NB: faces are managed by qSide_c, we only store copies here
  std::vector<qFace_c *> faces;

public:
  qNode_c(int _Zsplit) : z_splitter(_Zsplit), z(0),
                          x(0), y(0), dx(0), dy(0),
                          front_l(NULL), front_n(NULL),
                          back_l(NULL),  back_n(NULL),
                          min_x(0), min_y(0),
                          max_x(0), max_y(0),
                          faces()
  { }

  ~qNode_c()
  {
    if (front_l) delete front_l;
    if (front_n) delete front_n;

    if (back_l) delete back_l;
    if (back_n) delete back_n;
  }

  void Flip()
  {
    if (z_splitter)
      z_splitter = 3 - z_splitter;

    qLeaf_c *tmp_l = front_l; front_l = back_l; back_l = tmp_l;
    qNode_c *tmp_n = front_n; front_n = back_n; back_n = tmp_n;

    dx = -dx;
    dy = -dy;
  }

  void AssignFaces(qSide_c *S)
  {
    SYS_ASSERT(S->seg); // should never be a portal

    for (unsigned int f = 0; f < S->faces.size(); f++)
    {
      faces.push_back(S->faces[f]);
    }
  }

  void BBoxFromChildren()
  {
    min_x = +9e5; max_x = -9e5;
    min_y = +9e5; max_x = -9e5;

    // TODO z coordinate

    if (front_l == SOLID_LEAF)
    {
      // do nothing (???)
    }
    else if (front_l)
    {
      min_x = MIN(min_x, front_l->min_x);
      min_y = MIN(min_y, front_l->min_y);
      max_x = MAX(max_x, front_l->max_x);
      max_y = MAX(max_y, front_l->max_y);
    }
    else
    {
      SYS_ASSERT(front_n);

      min_x = MIN(min_x, front_n->min_x);
      min_y = MIN(min_y, front_n->min_y);
      max_x = MAX(max_x, front_n->max_x);
      max_y = MAX(max_y, front_n->max_y);
    }

    if (back_l == SOLID_LEAF)
    {
      // do nothing (???)
    }
    else if (back_l)
    {
      min_x = MIN(min_x, back_l->min_x);
      min_y = MIN(min_y, back_l->min_y);
      max_x = MAX(max_x, back_l->max_x);
      max_y = MAX(max_y, back_l->max_y);
    }
    else
    {
      SYS_ASSERT(back_n);

      min_x = MIN(min_x, back_n->min_x);
      min_y = MIN(min_y, back_n->min_y);
      max_x = MAX(max_x, back_n->max_x);
      max_y = MAX(max_y, back_n->max_y);
    }

    SYS_ASSERT(min_x <= max_x);
    SYS_ASSERT(min_y <= max_y);
  }
};


void DoAssignFaces(qNode_c *N, qSide_c *S)
{
  N->AssignFaces(S);
}



//------------------------------------------------------------------------

static double EvaluatePartition(qLeaf_c *leaf, qSide_c *part)
{
  int back   = 0;
  int front  = 0;
  int splits = 0;

  double pdx = part->x2 - part->x1;
  double pdy = part->y2 - part->y1;

  std::list<qSide_c *>::iterator SI;

  for (SI = leaf->sides.begin(); SI != leaf->sides.end(); SI++)
  {
    qSide_c *S = *SI;

    // FIXME !!!! TEMP HACK ignoring portals
    if (! S->seg)
      continue;

    // get state of lines' relation to each other
    double a = PerpDist(S->x1, S->y1,
                        part->x1, part->y1, part->x2, part->y2);

    double b = PerpDist(S->x2, S->y2,
                        part->x1, part->y1, part->x2, part->y2);

    double fa = fabs(a);
    double fb = fabs(b);

    if (fa <= Q_EPSILON && fb <= Q_EPSILON)
    {
      // lines are colinear

      double sdx = S->x2 - S->x1;
      double sdy = S->y2 - S->y1;

      if (pdx * sdx + pdy * sdy < 0.0)
        back++;
      else
        front++;

      continue;
    }

    if (fa <= Q_EPSILON || fb <= Q_EPSILON)
    {
      // partition passes through one vertex

      if ( ((fa <= Q_EPSILON) ? b : a) >= 0 )
        front++;
      else
        back++;

      continue;
    }

    if (a > 0 && b > 0)
    {
      front++;
      continue;
    }

    if (a < 0 && b < 0)
    {
      back++;
      continue;
    }

    // the partition line will split it

    splits++;

    back++;
    front++;
  }

///fprintf(stderr, "PARTITION CANDIDATE (%1.0f %1.0f)..(%1.0f %1.0f) : %d|%d splits:%d\n",
///        part->x1, part->y1, part->x2, part->y2,
///        back, front, splits);


  if (front == 0 || back == 0)
    return -1;

  // calculate heuristic
  int diff = ABS(front - back);

  double cost = (splits * (splits+1) * 365.0 + diff * 100.0) /
                (double)(front + back);

  // slight preference for axis-aligned planes
  if (! (fabs(pdx) < EPSILON || fabs(pdy) < EPSILON))
    cost += 1.0;

  return cost;
}


static qSide_c * FindPartition(qLeaf_c *leaf)
{
  std::list<qSide_c *>::iterator SI;

  double   best_c = 9e30;
  qSide_c *best_p = NULL;

  int count = 0;

  for (SI = leaf->sides.begin(); SI != leaf->sides.end(); SI++)
  {
    qSide_c *part = *SI;

    // ignore portal sides
    if (! part->seg)
      continue;

    count++;

    // TODO: Optimise for two-sided segments by skipping the back one

    // TODO: skip sides that lie on the same vertical plane

    double cost = EvaluatePartition(leaf, part);

///fprintf(stderr, "--> COST:%1.2f for %p\n", cost, part);

    if (cost < 0)  // not a potential candidate
      continue;

    if (! best_p || cost < best_c)
    {
      best_c = cost;
      best_p = part;
    }
  }
fprintf(stderr, "ALL DONE : best_c=%1.0f best_p=%p\n",
        best_p ? best_c : -9999, best_p);

  return best_p;
}

static void SplitDiagonalSides(qLeaf_c *L)
{
  // leafs are already split to be small than 240 units in width
  // and height.  If diagonal sides (the faces on them) are texture
  // mapped by an axis aligned plane, then we don't need to split
  // them.  Otherwise it is possible that the side is too long and
  // must be split.

  std::list<qSide_c *> new_bits;

  std::list<qSide_c *>::iterator SI;

  for (SI = L->sides.begin(); SI != L->sides.end(); SI++)
  {
    qSide_c *S = *SI;

    if (S->Length() > FACE_MAX_SIZE)
    {
fprintf(stderr, "Splitting DIAGONAL side %p length:%1.0f\n", S, S->Length());

      double ix = (S->x1 + S->x2) / 2.0;
      double iy = (S->y1 + S->y2) / 2.0;

      qSide_c *T = S->SplitAt(ix, iy);

      new_bits.push_back(T);
    }
  }

  while (! new_bits.empty())
  {
    L->sides.push_back(new_bits.back());

    new_bits.pop_back();
  }
}


class intersection_c
{
public:
  enum closed_kind_e
  {
    OPEN   = 0,
    BEFORE = 1,
    AFTER  = 2,
    CLOSED = 3
  };

  int closed;

  double along;  // distance along partition

  double x, y;

public:
   intersection_c() { }
  ~intersection_c() { }
};

struct Compare_Intersection_pred
{
  inline bool operator() (const intersection_c *A, const intersection_c *B) const
  {
    return A->along < B->along;
  }
};

static void AddIntersection(std::vector<intersection_c *>& cut_list,
                            qNode_c *part, double x, double y,
                            int closed = intersection_c::OPEN)
{
  intersection_c *K = new intersection_c();

  K->closed = closed;
  K->x = x;
  K->y = y;

  K->along = AlongDist(x, y,
                       part->x, part->y,
                       part->x + part->dx, part->y + part->dy);

  cut_list.push_back(K);
}

#if 0
static void DumpIntersections(std::vector<intersection_c *>& cut_list)
{
  static const char *closed_names[4] =
  {
    "OPEN", "BEFORE", "AFTER", "CLOSED"
  };

  fprintf(stderr, "INTERSECTIONS:\n");

  for (unsigned int i = 0; i < cut_list.size(); i++)
  {
    intersection_c *K = cut_list[i];

fprintf(stderr, "(%1.1f %1.1f) along:%8.3f closed:%s\n",
        K->x, K->y, K->along,
        (K->closed < 0 || K->closed > 4) ? "?????" :
        closed_names[K->closed]);
  }

fprintf(stderr, "\n");
}
#endif

static void MergeIntersections(std::vector<intersection_c *>& cut_list)
{
  // sort intersections by their position on the partition line,
  // and merge any points at the same location.

  std::sort(cut_list.begin(), cut_list.end(), Compare_Intersection_pred());

///  DumpIntersections(cut_list);

  unsigned int pos = 0;

  while (pos < cut_list.size())
  {
    unsigned int n = 0;  // neighbour count

    double along = cut_list[pos]->along;

    while ( (pos + n + 1) < cut_list.size() &&
            (cut_list[pos+n+1]->along - along) < Q_EPSILON )
    { n++; }

    for (unsigned int i = 1; i <= n; i++)
    {
      cut_list[pos]->closed |= cut_list[pos+i]->closed;

      delete cut_list[pos+i];
      cut_list[pos+i] = NULL;
    }

    pos += (n + 1);
  }

  // remove the NULL pointers
  std::vector<intersection_c *>::iterator ENDP;
  ENDP = std::remove(cut_list.begin(), cut_list.end(), (intersection_c*)NULL);
  cut_list.erase(ENDP, cut_list.end());

///  DumpIntersections(cut_list);
}

static void CreatePortals(std::vector<intersection_c *>& cut_list, 
                          qNode_c *part, qLeaf_c *front_l, qLeaf_c *back_l)
{
  for (unsigned int i = 0; i < cut_list.size()-1; i++)
  {
    intersection_c *K1 = cut_list[i];
    intersection_c *K2 = cut_list[i+1];

    bool k1_open = (K1->closed & intersection_c::AFTER)  ? false : true;
    bool k2_open = (K2->closed & intersection_c::BEFORE) ? false : true;

    if (k1_open != k2_open) // sanity check
    {
      fprintf(stderr, "WARNING: portal mismatch from (%1.1f %1.1f) to (%1.1f %1.1f)\n",
              K1->x, K1->y, K2->x, K2->y);
      continue;
    }

    if (! (k1_open && k2_open))
      continue;

    if (K2->along - K1->along < 0.99) // don't create tiny portals
      continue;

    // check if portal crosses solid space
    // (NOTE: this is hackish!)
    double mx = (K1->x + K2->x) / 2.0;
    double my = (K1->y + K2->y) / 2.0;

    if (CSG2_PointInSolid(mx, my))
      continue;

    qSide_c *front_pt = qSide_c::NewPortal(K1->x,K1->y, K2->x,K2->y, 0);
    qSide_c * back_pt = qSide_c::NewPortal(K1->x,K1->y, K2->x,K2->y, 1);

    front_l->sides.push_back(front_pt);
     back_l->sides.push_back( back_pt);

#if 0
    fprintf(stderr, "Portal along (%1.1f %1.1f) .. (%1.1f %1.1f)\n",
         K1->x,K1->y, K2->x,K2->y);
#endif
  }
}

static void DumpLeaf(qLeaf_c *L)
{
  std::list<qSide_c *>::iterator SI;

fprintf(stderr, "LEAF %p\n{\n", L);
  for (SI = L->sides.begin(); SI != L->sides.end(); SI++)
  {
    qSide_c *S = *SI;
    
    if (S->seg)
      fprintf(stderr, "  side #%p : seg:%p side:%d (%1.0f,%1.0f)..(%1.0f,%1.0f) \n",
              S, S->seg, S->side, S->x1, S->y1, S->x2, S->y2);
    else
      fprintf(stderr, "  side #%p : PORTAL side:%d (%1.0f,%1.0f)..(%1.0f,%1.0f) \n",
              S, S->side, S->x1, S->y1, S->x2, S->y2);
  }
fprintf(stderr, "}\n");
}

static void Split_XY(qNode_c *part, qLeaf_c *front_l, qLeaf_c *back_l)
{
  std::list<qSide_c *> all_sides;

  all_sides.swap(front_l->sides);

  std::vector<intersection_c *> cut_list;


  while (! all_sides.empty())
  {
    qSide_c *S = all_sides.front();

    all_sides.pop_front();

    double sdx = S->x2 - S->x1;
    double sdy = S->y2 - S->y1;

    // get state of lines' relation to each other
    double a = PerpDist(S->x1, S->y1,
                        part->x, part->y,
                        part->x + part->dx, part->y + part->dy);

    double b = PerpDist(S->x2, S->y2,
                        part->x, part->y,
                        part->x + part->dx, part->y + part->dy);

    double fa = fabs(a);
    double fb = fabs(b);

    if (fa <= Q_EPSILON && fb <= Q_EPSILON)
    {
      // lines are colinear

      if (part->dx * sdx + part->dy * sdy < 0.0)
      {
        back_l->sides.push_back(S);

        AddIntersection(cut_list, part, S->x2, S->y2, intersection_c::AFTER);
        AddIntersection(cut_list, part, S->x1, S->y1, intersection_c::BEFORE);
      }
      else
      {
        front_l->sides.push_back(S);

        AddIntersection(cut_list, part, S->x1, S->y1, intersection_c::AFTER);
        AddIntersection(cut_list, part, S->x2, S->y2, intersection_c::BEFORE);
      }

      // remember the faces along this node
      part->AssignFaces(S);

      S->on_node = part;
      continue;
    }

    if (fa <= Q_EPSILON || fb <= Q_EPSILON)
    {
      // partition passes through one vertex

      if ( ((fa <= Q_EPSILON) ? b : a) >= 0 )
        front_l->sides.push_back(S);
      else
        back_l->sides.push_back(S);

      if (fa <= Q_EPSILON)
        AddIntersection(cut_list, part, S->x1, S->y1);
      else // fb <= Q_EPSILON
        AddIntersection(cut_list, part, S->x2, S->y2);

      continue;
    }

    if (a > 0 && b > 0)
    {
      front_l->sides.push_back(S);
      continue;
    }

    if (a < 0 && b < 0)
    {
      back_l->sides.push_back(S);
      continue;
    }

    /* need to split it */

    // determine the intersection point
    double along = a / (a - b);

    double ix = S->x1 + along * sdx;
    double iy = S->y1 + along * sdy;

    qSide_c *T = S->SplitAt(ix, iy);

    if (a < 0)
    {
       back_l->sides.push_back(S);
      front_l->sides.push_back(T);
    }
    else
    {
      SYS_ASSERT(b < 0);

      front_l->sides.push_back(S);
       back_l->sides.push_back(T);
    }

    AddIntersection(cut_list, part, ix, iy);
  }

  MergeIntersections(cut_list);

  CreatePortals(cut_list, part, front_l, back_l);

if (0) {
for (int kk=0; kk < 2; kk++)
{
  fprintf(stderr, "%s:\n", (kk==0) ? "FRONT" : "BACK");
  DumpLeaf((kk==0) ? front_l : back_l);
}}

}


static void Partition_Solid(qLeaf_c *leaf, qNode_c ** out_n, qLeaf_c ** out_l)
{
  // handle sides first

  std::list<qSide_c *>::iterator SI;

  for (SI = leaf->sides.begin(); SI != leaf->sides.end(); SI++)
  {
    qSide_c *S = *SI;

    if (S->seg && ! S->on_node)
    {
      qNode_c * node = new qNode_c(0 /* z_splitter */);

      node->x = S->x1;
      node->y = S->y1;

      node->dx = S->x2 - S->x1;
      node->dy = S->y2 - S->y1;

      // find _ALL_ sides that lie on the partition
      std::list<qSide_c *>::iterator TI;

      for (TI = leaf->sides.begin(); TI != leaf->sides.end(); TI++)
      {
        qSide_c *T = *TI;

        if (! T->seg || T->on_node)
          continue;

        double a = PerpDist(T->x1, T->y1,  S->x1, S->y1, S->x2, S->y2);
        double b = PerpDist(T->x2, T->y2,  S->x1, S->y1, S->x2, S->y2);

        if (! (fabs(a) <= Q_EPSILON && fabs(b) <= Q_EPSILON))
          continue;

        node->AssignFaces(T);

        T->on_node = node;
      }

      node->back_l = SOLID_LEAF;

      Partition_Solid(leaf, &node->front_n, &node->front_l);

      node->BBoxFromChildren();

      (*out_n) = node;
      return;
    }
  }


  merge_gap_c *gap = leaf->GetGap();

  if (! leaf->ceil_on_node)
  {
      leaf->ceil_on_node = true;

      qNode_c * node = new qNode_c(2 /* z_splitter */);

      node->z = gap->GetZ2();

      SYS_ASSERT(leaf->ceil);
      node->faces.push_back(leaf->ceil);

      node->back_l = SOLID_LEAF;

      Partition_Solid(leaf, &node->front_n, &node->front_l);

      node->BBoxFromChildren();

      (*out_n) = node;
      return;
  }


  SYS_ASSERT(! leaf->floor_on_node);
  {
      leaf->floor_on_node = true;
  
      qNode_c * node = new qNode_c(1 /* z_splitter */);

      node->z = gap->GetZ1();

      SYS_ASSERT(leaf->floor);
      node->faces.push_back(leaf->floor);

      // End of the road, folks!
      node->front_l = leaf;
      node-> back_l = SOLID_LEAF;

      node->BBoxFromChildren();

      (*out_n) = node;
      return;
  }
}

static void Partition_Z(qLeaf_c *leaf, qNode_c ** out_n, qLeaf_c ** out_l)
{
  merge_region_c *R = leaf->GetRegion();

  if (leaf->numgap > 1)
  {
    int new_g = leaf->gap + leaf->numgap / 2;

    qLeaf_c *top_leaf = new qLeaf_c(*leaf, new_g);

    // TODO: OPTIMISE THIS : too many nodes!  Use top of gaps[new_g-1] as
    //       the splitting plane.

    qNode_c *node = new qNode_c(1 /* z_splitter */);

    // choose height halfway between the two gaps (in the solid)
    node->z = (R->gaps[new_g-1]->GetZ2() + R->gaps[new_g]->GetZ1()) / 2.0;

    top_leaf->numgap = leaf->gap + leaf->numgap - new_g;
        leaf->numgap = new_g - leaf->gap;

    Partition_Z(top_leaf, &node->front_n, &node->front_l);
    Partition_Z(    leaf, &node->back_n,  &node->back_l);

    node->BBoxFromChildren();

    *out_n = node;
    return;
  }

  SYS_ASSERT(leaf->numgap == 1);

  // create face list for the leaf
  leaf->AssignFaces();


  // create floor and ceiling faces here
  leaf->floor = new qFace_c(qFace_c::FLOOR);
  leaf->ceil  = new qFace_c(qFace_c::CEIL);

  leaf->floor->gap = leaf->gap;
  leaf-> ceil->gap = leaf->gap;

  leaf->floor->floor_leaf = leaf;
  leaf-> ceil->floor_leaf = leaf;


  Partition_Solid(leaf, out_n, out_l);
}


static void Partition_XY(qLeaf_c *leaf, qNode_c **out_n, qLeaf_c **out_l)
{
  bool is_root = (out_l == NULL);

  SYS_ASSERT(out_n);

  qSide_c *best_p = FindPartition(leaf);
  qNode_c *node;

  if (! best_p)
  {
    // current leaf is convex
///fprintf(stderr, "LEAF %p IS CONVEX\n", leaf);

    leaf->ComputeBBox();
    
    bool too_big = false;
    
    // faces must not be too large because of the way the Quake
    // texture mapping works.  Here we are abusing the node
    // builder to ensure floor and ceiling faces are OK.
    double l_width  = leaf->max_x - leaf->min_x;
    double l_height = leaf->max_y - leaf->min_y;

    if (l_width > FACE_MAX_SIZE || l_height > FACE_MAX_SIZE)
    {
///fprintf(stderr, "__ BUT TOO BIG: %1.0f x %1.0f\n", l_width , l_height);

      too_big = true;
    }

    // we need a root node, even on the simplest possible map.

    if (! is_root && ! too_big)
    {
      leaf->numgap = (int) leaf->GetRegion()->gaps.size();
      SYS_ASSERT(leaf->numgap > 0);

      SplitDiagonalSides(leaf);

      Partition_Z(leaf, out_n, out_l);
      return;
    }

    node = new qNode_c(0 /* z_splitter */);

    if (l_width > l_height)
    {
      // vertical divider
      node->x = (leaf->min_x + leaf->max_x) / 2.0;
      node->y = leaf->min_y;

      node->dx = 0;
      node->dy = leaf->max_y - leaf->min_y;
    }
    else // horizontal divider
    {
      node->x = leaf->min_x;
      node->y = (leaf->min_y + leaf->max_y) / 2.0;

      node->dx = leaf->max_x - leaf->min_x;
      node->dy = 0;
    }
  }
  else
  {
// fprintf(stderr, "LEAF HAS SPLITTER %p \n", best_p);
    node = new qNode_c(0 /* z_splitter */);

    node->x = best_p->x1;
    node->y = best_p->y1;

    node->dx = best_p->x2 - node->x;
    node->dy = best_p->y2 - node->y;
  }


fprintf(stderr, "Using partition (%1.0f,%1.0f) to (%1.2f,%1.2f)\n",
                 node->x, node->y,
                 node->x + node->dx, node->y + node->dy);

  qLeaf_c *front_l = leaf;
  qLeaf_c *back_l  = new qLeaf_c;

  Split_XY(node, leaf, back_l);


  Partition_XY(front_l, &node->front_n, &node->front_l);
  Partition_XY( back_l, &node-> back_n, &node-> back_l);

  node->BBoxFromChildren();

  *out_n = node;
}


static void DoAddFace(qSide_c *S, int gap, double z1, double z2)
{
  SYS_ASSERT(z2 > z1);

  // make sure face height does not exceed the limit
  if (z2 - z1 > FACE_MAX_SIZE)
  {
    int num = 1 + (int)floor((z2 - z1) / (double)FACE_MAX_SIZE);

///fprintf(stderr, "Splitting tall face (%1.0f .. %1.0f) into %d pieces\n", z1, z2, num);

    SYS_ASSERT(num >= 2);

    for (int i = 0; i < num; i++)
    {
      double nz1 = z1 + (z2 - z1) *  i    / (double)num;
      double nz2 = z1 + (z2 - z1) * (i+1) / (double)num;

      DoAddFace(S, gap, nz1, nz2);
    }

    return;
  }

  qFace_c *F = new qFace_c(qFace_c::WALL, gap, z1, z2);

  S->AddFace(F);
}

static void MakeSide(qLeaf_c *leaf, merge_segment_c *seg, int side)
{
  qSide_c *S = leaf->AddSide(seg, side);

  // create the faces
  merge_region_c *R  = (side == 0) ? seg->front : seg->back;
  merge_region_c *RX = (side == 0) ? seg->back  : seg->front;

  for (unsigned int k = 0; k < R->gaps.size(); k++)
  {
    merge_gap_c *G = R->gaps[k];

    double gz1 = G->GetZ1();
    double gz2 = G->GetZ2();

    // simple case: other side is completely solid
    if (RX == NULL || RX->gaps.size() == 0)
    {
      DoAddFace(S, k, gz1, gz2);

///fprintf(stderr, "Making face %1.0f..%1.0f gap:%u on one-sided line (%1.0f,%1.0f) - (%1.0f,%1.0f)\n",
///        gz1, gz2, k, S->x1, S->y1, S->x2, S->y2);
      continue;
    }

    // complex case: compare with solids on other side

    for (unsigned m = 0; m <= RX->gaps.size(); m++)
    {
      double sz1 = (m == 0) ? -9e6 : RX->gaps[m-1]->GetZ2();
      double sz2 = (m == RX->gaps.size()) ? +9e6 : RX->gaps[m]->GetZ1();

      if (sz1 < gz1) sz1 = gz1;
      if (sz2 > gz2) sz2 = gz2;

      if (sz2 > sz1 + 0.99)  // don't create tiny faces
      {
        DoAddFace(S, k, sz1, sz2);

///fprintf(stderr, "Making face %1.0f..%1.0f gap:%u neighbour:%u (%1.0f,%1.0f) - (%1.0f,%1.0f) side:%d\n",
///        sz1, sz2, k, m, S->x1, S->y1, S->x2, S->y2, side);
      }
    }
  }
}


//------------------------------------------------------------------------


void Q1_BuildBSP( void )
{
  // INPUTS:
  //   all the stuff created by CSG_MergeAreas
  //
  // OUTPUTS:
  //   a BSP tree consisting of nodes, leaves and faces

  // ALGORITHM:
  //   1. create a qSide list from every segment
  //   2. while list is not yet convex:
  //      (a) find a splitter side --> create qNode
  //      (b) split list into front and back
  //      (c) recursively handle front/back lists
  //   3. perform Z splitting (the gaps)
  //   4. perform solid splitting

  SOLID_LEAF = new qLeaf_c();
  SOLID_LEAF->contents = CONTENTS_SOLID;

  qLeaf_c *begin = new qLeaf_c();

  for (unsigned int i = 0; i < mug_segments.size(); i++)
  {
    merge_segment_c *S = mug_segments[i];

    if (S->front && S->front->gaps.size() > 0)
      MakeSide(begin, S, 0);

    if (S->back && S->back->gaps.size() > 0)
      MakeSide(begin, S, 1);
  }

  // NOTE WELL: we assume at least one partition (hence at least
  //            one node).  The simplest possible map is already a
  //            convex space (no partitions are needed) so in that
  //            case we use an arbitrary splitter plane.

fprintf(stderr, "Quake1_BuildBSP BEGUN\n");
  Partition_XY(begin, &Q_ROOT, NULL);
}


//------------------------------------------------------------------------

static dmodel_t model;

static qLump_c *q_nodes;
static qLump_c *q_leafs;
static qLump_c *q_faces;
static qLump_c *q_mark_surfs;
static qLump_c *q_surf_edges;
static qLump_c *q_clip_nodes;

static int total_nodes;
static int total_mark_surfs;
static int total_surf_edges;


static void DoAddEdge(double x1, double y1, double z1,
                      double x2, double y2, double z2,
                      dface_t *face, dleaf_t *raw_lf = NULL)
{
  u16_t v1 = BSP_AddVertex(x1, y1, z1);
  u16_t v2 = BSP_AddVertex(x2, y2, z2);

  if (v1 == v2)
  {
    Main_FatalError("INTERNAL ERROR (Q1 AddEdge): zero length!\n"
                    "coordinate (%1.2f %1.2f %1.2f)\n", x1, y1, z1);
  }

  s32_t edge_idx = BSP_AddEdge(v1, v2);


  edge_idx = LE_S32(edge_idx);

  q_surf_edges->Append(&edge_idx, 4);

  total_surf_edges += 1;

  face->numedges += 1;


  // update bounding boxes
  double lows[3], highs[3];

  lows[0] = MIN(x1, x2);  highs[0] = MAX(x1, x2);
  lows[1] = MIN(y1, y2);  highs[1] = MAX(y1, y2);
  lows[2] = MIN(z1, z2);  highs[2] = MAX(z1, z2);

  for (int b = 0; b < 3; b++)
  {
    s16_t low  =  (I_ROUND( lows[b]) - 2) & ~3;
    s16_t high = ((I_ROUND(highs[b]) + 2) |  3) + 1;

    if (raw_lf)
    {
      raw_lf->mins[b] = MIN(raw_lf->mins[b], low);
      raw_lf->maxs[b] = MAX(raw_lf->maxs[b], high);
    }
  }
}


static void DoAddSurf(u16_t index, dleaf_t *raw_lf )
{
  index = LE_U16(index);

  q_mark_surfs->Append(&index, 2);

  total_mark_surfs += 1;

  raw_lf->num_marksurf += 1;
}


struct Compare_FloorAngle_pred
{
  double *angles;

   Compare_FloorAngle_pred(double *p) : angles(p) { }
  ~Compare_FloorAngle_pred() { }

  inline bool operator() (int A, int B) const
  {
    return angles[A] < angles[B];
  }
};


static int CollectClockwiseVerts(float *vert_x, float *vert_y, qLeaf_c *leaf, bool anticlock)
{
  int v_num = 0;

  std::list<qSide_c *>::iterator SI;

  double mid_x = 0;
  double mid_y = 0;
  

  for (SI = leaf->sides.begin(); SI != leaf->sides.end(); SI++, v_num++)
  {
    qSide_c *S = *SI;

    vert_x[v_num] = S->x1;
    vert_y[v_num] = S->y1;

    mid_x += vert_x[v_num];
    mid_y += vert_y[v_num];
  }

  mid_x /= v_num;
  mid_y /= v_num;

  
  // determine angles, then sort into clockwise order

  double angles[256];

  std::vector<int> mapping(v_num);

  for (int a = 0; a < v_num; a++)
  {
    angles[a] = CalcAngle(mid_x, mid_y, vert_x[a], vert_y[a]);

    if (! anticlock)
      angles[a] *= -1.0;

    mapping[a] = a;
  }


  std::sort(mapping.begin(), mapping.end(),
            Compare_FloorAngle_pred(angles));


  // apply mapping to vertices
  float old_x[256];
  float old_y[256];

  for (int k = 0; k < v_num; k++)
  {
    old_x[k] = vert_x[k];
    old_y[k] = vert_y[k];
  }

///fprintf(stderr, "\nMIDDLE @ (%1.2f %1.2f) COUNT:%d\n", mid_x, mid_y, v_num);
  for (int m = 0; m < v_num; m++)
  {
    vert_x[m] = old_x[mapping[m]];
    vert_y[m] = old_y[mapping[m]];

///fprintf(stderr, "___ (%+5.0f %+5.0f)\n", vert_x[m], vert_y[m]);
  }

  return v_num;
}


static csg_brush_c * PolyForSideTexture(merge_region_c *R, double z1, double z2)
{
  // find the brush which we will use for the side texture
  // FIXME: duplicate code in dm_level : make one good function

  csg_brush_c *MID = NULL;
  double best_h = 0;

  for (unsigned int j = 0; j < R->brushes.size(); j++)
  {
    csg_brush_c *A = R->brushes[j];

    if (A->z2 < z1 + EPSILON)
      continue;

    if (A->z1 > z2 - EPSILON)
      continue;

    {
      double h = A->z2 - A->z1;

      // TODO: priorities

//      if (MID && fabs(h - best_h) < EPSILON)
//      { /* same height, prioritise */ }

      if (h > best_h)
      {
        best_h = h;
        MID = A;
      }
    }
  }

  return MID;
}


static int CalcTextureFlag(const char *tex_name)
{
  if (tex_name[0] == '*')
    return TEX_SPECIAL;

  if (strncmp(tex_name, "sky", 3) == 0)
    return TEX_SPECIAL;

  return 0;
}

static double DotProduct3(const double *A, const double *B)
{
  return A[0] * B[0] + A[1] * B[1] + A[2] * B[2];
}

static void GetExtents(double min_s, double min_t, double max_s, double max_t,
                       int *ext_W, int *ext_H)
{
  // -AJA- this matches the logic in the Quake1 engine.

  int bmin_s = (int)floor(min_s / 16.0);
  int bmin_t = (int)floor(min_t / 16.0);

  int bmax_s = (int)ceil(max_s / 16.0);
  int bmax_t = (int)ceil(max_t / 16.0);

  *ext_W = bmax_s - bmin_s + 1;
  *ext_H = bmax_t - bmin_t + 1;
}

static void MakeFloorFace(qFace_c *F, dface_t *face)
{
  qLeaf_c *leaf = F->floor_leaf;
  SYS_ASSERT(leaf);

  merge_region_c *R = leaf->GetRegion();
  SYS_ASSERT(R);

  merge_gap_c *gap = R->gaps.at(F->gap);

  double z1 = gap->GetZ1();
  double z2 = gap->GetZ2();

  double z = (F->kind == qFace_c::CEIL) ? z2 : z1;
///fprintf(stderr, "MakeFloorFace: F=%p kind:%d @ z:%1.0f\n", F, F->kind, z);


  bool is_ceil = (F->kind == qFace_c::CEIL) ? true : false;
  bool flipped;

  face->planenum = BSP_AddPlane(0, 0, z,
                                0, 0, is_ceil ? -1 : +1, &flipped);

  face->side = flipped ? 1 : 0;

  const char *texture = is_ceil ? gap->CeilTex() : gap->FloorTex();

  int flags = CalcTextureFlag(texture);

  double s[4] = { 1.0, 0.0, 0.0, 0.0 };
  double t[4] = { 0.0, 1.0, 0.0, 0.0 };

  face->texinfo = Q1_AddTexInfo(texture, flags, s, t);

  face->styles[0] = 0xFF;  // no lightmap
  face->styles[1] = 0xFF;
  face->styles[2] = 0xFF;
  face->styles[3] = 0xFF;

  face->lightofs = -1;  // no lightmap

  // collect the vertices and sort in clockwise order

  float vert_x[256];
  float vert_y[256];

  int v_num = CollectClockwiseVerts(vert_x, vert_y, leaf, flipped);

  double min_x = +9e9; double max_x = -9e9;
  double min_y = +9e9; double max_y = -9e9;


  // add the edges

  face->firstedge = total_surf_edges;
  face->numedges  = 0;

  for (int pos = 0; pos < v_num; pos++)
  {
    int p2 = (pos + 1) % v_num;

    DoAddEdge(vert_x[pos], vert_y[pos], z,
              vert_x[p2 ], vert_y[p2 ], z, face);

    min_x = MIN(min_x, vert_x[pos]);
    min_y = MIN(min_y, vert_y[pos]);
    max_x = MAX(max_x, vert_x[pos]);
    max_y = MAX(max_y, vert_y[pos]);
  }


  if (! (flags & TEX_SPECIAL))
  {
    int ext_W, ext_H;

    GetExtents(min_x, min_y, max_x, max_y, &ext_W, &ext_H);

    static int foo; foo++;
    face->styles[0] = 0; // (foo & 3); //!!!!!

    face->lightofs = 100; //!!!! Quake1_LightAddBlock(ext_W, ext_H, rand()&0x7F);
  }
}

static void MakeWallFace(qFace_c *F, dface_t *face)
{
  qSide_c *S = F->side;

  merge_region_c *R = S->GetRegion();
  SYS_ASSERT(R);

///  merge_gap_c *gap = R->gaps.at(F->gap);

  double z1 = F->z1;
  double z2 = F->z2;


  bool flipped;

  face->planenum = BSP_AddPlane(S->x1, S->y1, 0,
                                (S->y2 - S->y1), (S->x1 - S->x2), 0,
                                &flipped);

  face->side = flipped ? 1 : 0;
  
  const char *texture = "error";

  merge_region_c *BACK = (S->side == 0) ? S->seg->back : S->seg->front;
///fprintf(stderr, "BACK = %p\n", BACK);
  if (BACK)
  {
    csg_brush_c *MID = PolyForSideTexture(BACK, z1, z2);
    if (MID)
      texture = MID->w_face->tex.c_str();
  }

  int flags = CalcTextureFlag(texture);

  double s[4] = { 0.0, 0.0, 0.0, 0.0 };
  double t[4] = { 0.0, 0.0, 1.0, 0.0 };

  if (fabs(S->x1 - S->x2) > fabs(S->y1 - S->y2))
  {
    s[0] = 1.0;
  }
  else
  {
    s[1] = 1.0;
  }

  face->texinfo = Q1_AddTexInfo(texture, flags, s, t);

  face->styles[0] = 0xFF;  // no lightmap
  face->styles[1] = 0xFF;
  face->styles[2] = 0xFF;
  face->styles[3] = 0xFF;

  face->lightofs = -1;  // no lightmap

  // add the edges

  face->firstedge = total_surf_edges;
  face->numedges  = 0;

  DoAddEdge(S->x1, S->y1, z1,  S->x1, S->y1, z2,  face);
  DoAddEdge(S->x1, S->y1, z2,  S->x2, S->y2, z2,  face);
  DoAddEdge(S->x2, S->y2, z2,  S->x2, S->y2, z1,  face);
  DoAddEdge(S->x2, S->y2, z1,  S->x1, S->y1, z1,  face);

  if (! (flags & TEX_SPECIAL))
  {
    double coord[4][3];

    coord[0][0] = S->x1; coord[0][1] = S->y1; coord[0][2] = z1;
    coord[1][0] = S->x1; coord[1][1] = S->y1; coord[1][2] = z2;
    coord[2][0] = S->x2; coord[2][1] = S->y2; coord[2][2] = z1;
    coord[3][0] = S->x2; coord[3][1] = S->y2; coord[3][2] = z2;

    double min_s = +9e9; double max_s = -9e9;
    double min_t = +9e9; double max_t = -9e9;

    for (int k = 0; k < 4; k++)
    {
      double ss = DotProduct3(s, coord[k]);
      double tt = DotProduct3(t, coord[k]);

      min_s = MIN(min_s, ss); max_s = MAX(max_s, ss);
      min_t = MIN(min_t, tt); max_t = MAX(max_t, tt);
    }

    int ext_W, ext_H;

    GetExtents(min_s, min_t, max_s, max_t, &ext_W, &ext_H);

    static int foo = 0; foo++;
    face->styles[0] = 0; // (foo & 3); //!!!!!

    face->lightofs = 100; //!!!! Quake1_LightAddBlock(ext_W, ext_H, 0x80|(rand()&0x7F));
  }
}

static void MakeFace(qFace_c *F, qNode_c *N)
{
  SYS_ASSERT(F->index < 0);


  dface_t face;

  if (F->kind == qFace_c::WALL)
    MakeWallFace(F, &face);
  else
    MakeFloorFace(F, &face);


  // FIXME: fix endianness in face

  u16_t index = model.numfaces++;

  if (index >= MAX_MAP_FACES)
    Main_FatalError("Quake1 build failure: exceeded limit of %d FACES\n",
                    MAX_MAP_FACES);

  F->index = (int)index;

  q_faces->Append(&face, sizeof(face));
}


static s16_t MakeLeaf(qLeaf_c *leaf, dnode_t *parent)
{
  if (leaf == SOLID_LEAF)
    return -1;


  dleaf_t raw_lf;

  raw_lf.contents = leaf->contents;
  raw_lf.visofs   = -1;  // no visibility info

  int b;

  raw_lf.mins[0] = I_ROUND(leaf->min_x)-16;
  raw_lf.mins[1] = I_ROUND(leaf->min_y)-16;
  raw_lf.mins[2] = -2000;  //!!!!

  raw_lf.maxs[0] = I_ROUND(leaf->max_x)+16;
  raw_lf.maxs[1] = I_ROUND(leaf->max_y)+16;
  raw_lf.maxs[2] = 2000;  //!!!!

  memset(raw_lf.ambient_level, 0, sizeof(raw_lf.ambient_level));

  raw_lf.first_marksurf = total_mark_surfs;
  raw_lf.num_marksurf   = 0;


  // make faces
  for (unsigned int n = 0; n < leaf->faces.size(); n++)
  {
    qFace_c *F = leaf->faces[n];

    // should have been in a node already
    SYS_ASSERT(F->index >= 0);

    DoAddSurf(F->index, &raw_lf);
  }

  // ???  probably should just put into leaf->faces array 
  SYS_ASSERT(leaf->floor->index >= 0);
  SYS_ASSERT(leaf-> ceil->index >= 0);

  DoAddSurf(leaf->floor->index, &raw_lf);
  DoAddSurf(leaf-> ceil->index, &raw_lf);


  for (b = 0; b < 3; b++)
  {
    parent->mins[b] = MIN(parent->mins[b], raw_lf.mins[b]);
    parent->maxs[b] = MAX(parent->maxs[b], raw_lf.maxs[b]);
  }


  // FIXME: fix endianness in raw_lf

  s32_t index = model.visleafs++;

  if (index >= MAX_MAP_LEAFS)
    Main_FatalError("Quake1 build failure: exceeded limit of %d LEAFS\n",
                    MAX_MAP_LEAFS);

  q_leafs->Append(&raw_lf, sizeof(raw_lf));

  return -(index+2);  // index+2 because the first leaf is SOLID
}


static s32_t RecursiveMakeNodes(qNode_c *node, dnode_t *parent)
{
  dnode_t raw_nd;

  int b;
  bool flipped;

  if (node->z_splitter)
    raw_nd.planenum = BSP_AddPlane(0, 0, node->z,
                                   0, 0, (node->z_splitter==2) ? -1 : +1, &flipped);
  else
    raw_nd.planenum = BSP_AddPlane(node->x, node->y, 0,
                                   node->dy, -node->dx, 0, &flipped);
  if (flipped)
    node->Flip();


  raw_nd.firstface = 0;
  raw_nd.numfaces  = 0;

  raw_nd.mins[0] = I_ROUND(node->min_x)-32;
  raw_nd.mins[1] = I_ROUND(node->min_y)-32;
  raw_nd.mins[2] = -2000;  //!!!!

  raw_nd.maxs[0] = I_ROUND(node->max_x)+32;
  raw_nd.maxs[1] = I_ROUND(node->max_y)+32;
  raw_nd.maxs[2] = 2000;  //!!!!


  // make faces [NOTE: must be done before recursing down]
  for (unsigned int j = 0; j < node->faces.size(); j++)
  {
    qFace_c *F = node->faces[j];

///fprintf(stderr, "node face: %p kind:%d (node %1.4f,%1.4f += %1.4f,%1.4f)\n",
///        F, F->kind, node->x, node->y, node->dx, node->dy);

    SYS_ASSERT(F->index < 0);

    MakeFace(F, node);

    SYS_ASSERT(F->index >= 0);

    if (j == 0)
      raw_nd.firstface = F->index;
    
    raw_nd.numfaces++;
  }


  if (node->front_n)
    raw_nd.children[0] = RecursiveMakeNodes(node->front_n, &raw_nd);
  else
    raw_nd.children[0] = MakeLeaf(node->front_l, &raw_nd);

  if (node->back_n)
    raw_nd.children[1] = RecursiveMakeNodes(node->back_n, &raw_nd);
  else
    raw_nd.children[1] = MakeLeaf(node->back_l, &raw_nd);


  if (parent)
  {
    for (b = 0; b < 3; b++)
    {
      parent->mins[b] = MIN(parent->mins[b], raw_nd.mins[b]);
      parent->maxs[b] = MAX(parent->maxs[b], raw_nd.maxs[b]);
    }
  }


  // -AJA- NOTE WELL: the Quake1 code assumes the root node is the
  //       very first one.  The following is a hack to achieve that.
  //       (Hopefully no other assumptions about node ordering exist
  //       in the Quake1 code!).

  if (! parent) // is_root
  {
    q_nodes->Prepend(&raw_nd, sizeof(raw_nd));

    return 0;
  }

  s32_t index = total_nodes++;

  if (index >= MAX_MAP_NODES)
    Main_FatalError("Quake1 build failure: exceeded limit of %d NODES\n",
                    MAX_MAP_NODES);

  // FIXME: fix endianness in raw_nd

  q_nodes->Append(&raw_nd, sizeof(raw_nd));

  return index;
}


static void CreateSolidLeaf(void)
{
  dleaf_t raw_lf;

  memset(&raw_lf, 0, sizeof(raw_lf));

  raw_lf.contents = CONTENTS_SOLID;
  raw_lf.visofs   = -1;  // no visibility info

  q_leafs->Append(&raw_lf, sizeof(raw_lf));
}


static void MapModel_Face(q1MapModel_c *model, int face, s16_t plane, bool flipped)
{
  dface_t raw_fc;

  raw_fc.planenum = plane;
  raw_fc.side = flipped ? 1 : 0;
 

  const char *texture = "error";

  double s[4] = { 0.0, 0.0, 0.0, 0.0 };
  double t[4] = { 0.0, 0.0, 0.0, 0.0 };

  if (face < 2)  // PLANE_X
  {
    s[1] = 1.0; t[2] = 1.0;

    texture = model->x_face->tex.c_str();
  }
  else if (face < 4)  // PLANE_Y
  {
    s[0] = 1.0; t[2] = 1.0;

    texture = model->y_face->tex.c_str();
  }
  else // PLANE_Z
  {
    s[0] = 1.0; t[1] = 1.0;

    texture = model->z_face->tex.c_str();
  }

  int flags = CalcTextureFlag(texture);

  raw_fc.texinfo = Q1_AddTexInfo(texture, flags, s, t);

  raw_fc.styles[0] = 0xFF;  // no lightmap
  raw_fc.styles[1] = 0xFF;
  raw_fc.styles[2] = 0xFF;
  raw_fc.styles[3] = 0xFF;

  raw_fc.lightofs = -1;  // no lightmap

  // add the edges

  raw_fc.firstedge = total_surf_edges;
  raw_fc.numedges  = 0;

  if (face < 2)  // PLANE_X
  {
    double x = (face==0) ? model->x1 : model->x2;
    double y1 = flipped  ? model->y2 : model->y1;
    double y2 = flipped  ? model->y1 : model->y2;

    // Note: this assumes the plane is positive
    DoAddEdge(x, y1, model->z1, x, y1, model->z2, &raw_fc);
    DoAddEdge(x, y1, model->z2, x, y2, model->z2, &raw_fc);
    DoAddEdge(x, y2, model->z2, x, y2, model->z1, &raw_fc);
    DoAddEdge(x, y2, model->z1, x, y1, model->z1, &raw_fc);
  }
  else if (face < 4)  // PLANE_Y
  {
    double y = (face==2) ? model->y1 : model->y2;
    double x1 = flipped  ? model->x1 : model->x2;
    double x2 = flipped  ? model->x2 : model->x1;

    DoAddEdge(x1, y, model->z1, x1, y, model->z2, &raw_fc);
    DoAddEdge(x1, y, model->z2, x2, y, model->z2, &raw_fc);
    DoAddEdge(x2, y, model->z2, x2, y, model->z1, &raw_fc);
    DoAddEdge(x2, y, model->z1, x1, y, model->z1, &raw_fc);
  }
  else // PLANE_Z
  {
    double z = (face==5) ? model->z1 : model->z2;
    double x1 = flipped  ? model->x2 : model->x1;
    double x2 = flipped  ? model->x1 : model->x2;

    DoAddEdge(x1, model->y1, z, x1, model->y2, z, &raw_fc);
    DoAddEdge(x1, model->y2, z, x2, model->y2, z, &raw_fc);
    DoAddEdge(x2, model->y2, z, x2, model->y1, z, &raw_fc);
    DoAddEdge(x2, model->y1, z, x1, model->y1, z, &raw_fc);
  }

  if (! (flags & TEX_SPECIAL))
  {
    static int foo = 0; foo++;
    raw_fc.styles[0] = (foo & 3);
    raw_fc.lightofs  = 100;  //!!! flat lighting index
  }

  q_faces->Append(&raw_fc, sizeof(raw_fc));
}

static void MapModel_Nodes(q1MapModel_c *model, int face_base, int leaf_base)
{
  model->nodes[0] = total_nodes;

  int mins[3], maxs[3];

  mins[0] = I_ROUND(model->x1)-32;
  mins[1] = I_ROUND(model->y1)-32;
  mins[2] = I_ROUND(model->z1)-64;

  maxs[0] = I_ROUND(model->x2)+32;
  maxs[1] = I_ROUND(model->y2)+32;
  maxs[2] = I_ROUND(model->z2)+64;

  for (int face = 0; face < 6; face++)
  {
    dnode_t raw_nd;
    dleaf_t raw_lf;

    double v;
    double dir;
    bool flipped;

    if (face < 2)  // PLANE_X
    {
      v = (face==0) ? model->x1 : model->x2;
      dir = (face==0) ? -1 : 1;
      raw_nd.planenum = BSP_AddPlane(v,0,0, dir,0,0, &flipped);
    }
    else if (face < 4)  // PLANE_Y
    {
      v = (face==2) ? model->y1 : model->y2;
      dir = (face==2) ? -1 : 1;
      raw_nd.planenum = BSP_AddPlane(0,v,0, 0,dir,0, &flipped);
    }
    else  // PLANE_Z
    {
      v = (face==5) ? model->z1 : model->z2;
      dir = (face==5) ? -1 : 1;
      raw_nd.planenum = BSP_AddPlane(0,0,v, 0,0,dir, &flipped);
    }

    raw_nd.children[0] = -(leaf_base + face + 2);
    raw_nd.children[1] = (face == 5) ? -1 : (model->nodes[0] + face + 1);

    if (flipped)
    {
      u16_t tmp = raw_nd.children[0];
      raw_nd.children[0] = raw_nd.children[1];
      raw_nd.children[1] = tmp;
    }

    raw_nd.firstface = face_base + face;
    raw_nd.numfaces  = 1;

    for (int i = 0; i < 3; i++)
    {
      raw_lf.mins[i] = raw_nd.mins[i] = mins[i];
      raw_lf.maxs[i] = raw_nd.maxs[i] = maxs[i];
    }

    raw_lf.contents = CONTENTS_EMPTY;
    raw_lf.visofs = -1;

    raw_lf.first_marksurf = total_mark_surfs;
    raw_lf.num_marksurf   = 0;

    memset(raw_lf.ambient_level, 0, sizeof(raw_lf.ambient_level));

    MapModel_Face(model, face, raw_nd.planenum, flipped);

    DoAddSurf(raw_lf.first_marksurf, &raw_lf);

    // TODO: fix endianness

    q_nodes->Append(&raw_nd, sizeof(raw_nd));
    q_leafs->Append(&raw_lf, sizeof(raw_lf));
  }
}


void Q1_CreateSubModels(qLump_c *L, int first_face, int first_leaf)
{
  for (unsigned int mm=0; mm < q1_all_mapmodels.size(); mm++)
  {
    q1MapModel_c *model = q1_all_mapmodels[mm];

    dmodel_t smod;

    smod.mins[0] = model->x1;  smod.maxs[0] = model->x2;
    smod.mins[1] = model->y1;  smod.maxs[1] = model->y2;
    smod.mins[2] = model->z1;  smod.maxs[2] = model->z2;

    smod.origin[0] = 0;
    smod.origin[1] = 0;
    smod.origin[2] = 0;

    smod.visleafs  = 6;
    smod.firstface = first_face;
    smod.numfaces  = 6;

    MapModel_Nodes(model, first_face, first_leaf);

    first_face  += 6;
    first_leaf  += 6;
    total_nodes += 6;

    for (int h = 0; h < 4; h++)
    {
      smod.headnode[h] = model->nodes[h];
    }

    // TODO: fix endianness in model
    L->Append(&smod, sizeof(smod));
  }
}


void Q1_CreateModel(void)
{
  qLump_c *lump = BSP_NewLump(LUMP_MODELS);

///  dmodel_t model;

  model.visleafs  = 0;
  model.firstface = 0;
  model.numfaces  = 0;

  total_nodes = 1;  // root node is always first

  q_nodes = BSP_NewLump(LUMP_NODES);
  q_leafs = BSP_NewLump(LUMP_LEAFS);
  q_faces = BSP_NewLump(LUMP_FACES);

  q_mark_surfs = BSP_NewLump(LUMP_MARKSURFACES);
  q_surf_edges = BSP_NewLump(LUMP_SURFEDGES);

  CreateSolidLeaf();

  RecursiveMakeNodes(Q_ROOT, NULL /* parent */);


  // set model bounding box
  double min_x, min_y, min_z;
  double max_x, max_y, max_z;

  CSG2_GetBounds(min_x, min_y, min_z,  max_x, max_y, max_z);

  model.mins[0] = min_x;  model.maxs[0] = max_x;
  model.mins[1] = min_y;  model.maxs[1] = max_y;
  model.mins[2] = min_z;  model.maxs[2] = max_z;

  model.origin[0] = 0;
  model.origin[1] = 0;
  model.origin[2] = 0;


  // clipping hulls
  q_clip_nodes = BSP_NewLump(LUMP_CLIPNODES);

  model.headnode[0] = 0; // root of drawing BSP
  model.headnode[1] = Q1_CreateClipHull(1, q_clip_nodes);
  model.headnode[2] = Q1_CreateClipHull(2, q_clip_nodes);
  model.headnode[3] = Q1_CreateClipHull(3, q_clip_nodes);


  // FIXME: fix endianness in model

  lump->Append(&model, sizeof(model));

  Q1_CreateSubModels(lump, model.numfaces, model.visleafs);
}

//--- editor settings ---
// vi:ts=2:sw=2:expandtab
